# Start Time Fix - Database Schema Update

## Problem Description

When a student submits an exam, the `start_time` field in the `student_exam_sessions` table was being automatically updated to the current timestamp, instead of remaining at the original exam start time.

### Root Cause

The database schema had an incorrect column definition:

```sql
`start_time` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp()
```

The `ON UPDATE current_timestamp()` clause causes MySQL to automatically update the `start_time` column whenever **ANY** field in the row is modified. This means when the exam submission updates `end_time`, `is_submitted`, and `total_score`, the `start_time` also gets updated unintentionally.

## Solution

Remove the `ON UPDATE current_timestamp()` clause from the `start_time` column definition.

### Corrected Schema

```sql
`start_time` timestamp NOT NULL DEFAULT current_timestamp()
```

Now the `start_time` will:
- ✅ Be set to the current timestamp when a new exam session is created
- ✅ Remain unchanged when the row is updated (e.g., on exam submission)
- ✅ Accurately reflect when the student actually started the exam

## Files Modified

1. **proctored_exam_app2.sql** (Line 172)
   - Updated the CREATE TABLE statement for fresh installations
   
2. **fix_start_time_column.sql** (NEW FILE)
   - Migration script for existing databases

## How to Apply This Fix

### For New Installations

If you're setting up a fresh database, simply import the updated `proctored_exam_app2.sql` file:

```bash
mysql -u root -p proctored_exam_app2 < proctored_exam_app2.sql
```

The correct schema will be applied automatically.

### For Existing Databases

If you already have a running database with exam data, you need to run the migration script:

#### Step 1: Backup Your Database

**IMPORTANT**: Always backup before making schema changes!

```bash
mysqldump -u root -p proctored_exam_app2 > backup_before_start_time_fix.sql
```

#### Step 2: Run the Migration Script

**Option A: Command Line**

```bash
mysql -u root -p proctored_exam_app2 < fix_start_time_column.sql
```

**Option B: phpMyAdmin**

1. Open phpMyAdmin
2. Select the `proctored_exam_app2` database
3. Click on the **SQL** tab
4. Copy and paste this command:

```sql
ALTER TABLE `student_exam_sessions` 
MODIFY COLUMN `start_time` timestamp NOT NULL DEFAULT current_timestamp();
```

5. Click **Go**

#### Step 3: Verify the Fix

Run this query to check the column definition:

```sql
SHOW CREATE TABLE student_exam_sessions;
```

Look for the `start_time` column. It should show:

```sql
`start_time` timestamp NOT NULL DEFAULT current_timestamp()
```

**WITHOUT** `ON UPDATE current_timestamp()` at the end.

## Impact Assessment

### What This Fixes

✅ **Accurate Time Tracking**: The time spent on exams will now be calculated correctly based on the actual start time

✅ **Admin Dashboard**: The "Time Spent" column in `admin/test_takers.php` will show accurate durations

✅ **Reporting**: Any reports or analytics based on exam duration will be correct

### Data Integrity

- **Existing Data**: This change does NOT affect existing data in the table
- **Future Records**: All new exam sessions will have the correct behavior
- **No Data Loss**: This is a schema-only change; no data is deleted or modified

### Backwards Compatibility

This fix is fully backwards compatible. The application code doesn't need any changes because:
- The PHP code never explicitly sets `start_time` on updates
- The column still has `DEFAULT current_timestamp()` for new records
- Only the unwanted auto-update behavior is removed

## Testing Recommendations

After applying the fix, test the following scenarios:

### Test 1: New Exam Session
1. Student starts a new exam
2. Note the start time in the database
3. Student submits the exam
4. **Verify**: start_time should remain unchanged

### Test 2: Time Calculation
1. Student starts exam at time T1
2. Student submits exam at time T2
3. Check admin/test_takers.php
4. **Verify**: Time spent = T2 - T1 (accurate duration)

### Test 3: Database Query
```sql
SELECT session_id, start_time, end_time, 
       TIMESTAMPDIFF(SECOND, start_time, end_time) as duration_seconds
FROM student_exam_sessions 
WHERE is_submitted = 1
ORDER BY session_id DESC
LIMIT 5;
```

**Verify**: The duration_seconds matches the actual time the student spent on the exam

## Technical Details

### MySQL Timestamp Behavior

MySQL timestamp columns support two special clauses:
- `DEFAULT current_timestamp()` - Sets the column value when a new row is inserted
- `ON UPDATE current_timestamp()` - Updates the column value whenever the row is modified

### Why This Was Wrong

The `start_time` should only be set once (when the exam session begins), not updated every time the row changes. The `ON UPDATE` clause was inappropriate for this use case.

### Correct Usage

- **start_time**: `DEFAULT current_timestamp()` only (set once, never auto-update)
- **updated_at**: `DEFAULT current_timestamp() ON UPDATE current_timestamp()` (tracks last modification)

## Rollback Procedure

If you need to rollback this change (not recommended):

```sql
ALTER TABLE `student_exam_sessions` 
MODIFY COLUMN `start_time` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp();
```

However, this will reintroduce the bug where start_time gets updated on exam submission.

## Support

If you encounter any issues after applying this fix:

1. Check that the ALTER TABLE command completed successfully
2. Verify the column definition with SHOW CREATE TABLE
3. Test with a new exam session
4. Check the application logs for any errors

## Summary

This is a critical fix that ensures accurate time tracking for exam sessions. The change is simple, safe, and should be applied to all installations of the proctored exam system.

**Priority**: High  
**Risk**: Low (schema-only change, no data modification)  
**Downtime**: None (ALTER TABLE is fast on small-medium tables)
