<?php
// admin/bulk_upload.php - Admin interface for bulk test-taker creation via CSV

require_once '../db.php';
require_admin();

// Set page title for header
$page_title = 'Bulk Upload Test-Takers';

$error = '';
$success = '';
$total_processed = 0;
$total_success = 0;
$total_failed = 0;
$failed_rows = [];

// Function to generate a random alphanumeric password (copied from create_user.php)
function generate_random_password($length = 10) {
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $password = '';
    for ($i = 0; $i < $length; $i++) {
        $password .= $characters[rand(0, strlen($characters) - 1)];
    }
    return $password;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['csv_file'])) {
    $file = $_FILES['csv_file'];

    if ($file['error'] !== UPLOAD_ERR_OK) {
        $error = 'File upload failed with error code: ' . $file['error'];
    } elseif ($file['type'] !== 'text/csv' && pathinfo($file['name'], PATHINFO_EXTENSION) !== 'csv') {
        $error = 'Invalid file type. Please upload a CSV file.';
    } else {
        $handle = fopen($file['tmp_name'], 'r');
        if ($handle === FALSE) {
            $error = 'Could not open the uploaded file.';
        } else {
            // Skip header row
            fgetcsv($handle);
            $row_number = 1;

            // Start transaction for bulk insert
            $mysqli->begin_transaction();
            $stmt = $mysqli->prepare("INSERT INTO users (username, password_hash, full_name, user_type, grade, staff_id, email) VALUES (?, ?, ?, 'Student', ?, ?, ?)");

            while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) {
                $row_number++;
                $total_processed++;

                // Expected CSV format: Full Name, Staff ID/Username, Email, Grade
                if (count($data) < 4) {
                    $failed_rows[] = "Row $row_number: Missing data fields.";
                    $total_failed++;
                    continue;
                }

                list($full_name, $username, $email, $grade) = array_map('trim', $data);
                $staff_id = $username; // Staff ID is now the Username

                if (empty($username) || empty($full_name) || empty($email) || empty($grade)) {
                    $failed_rows[] = "Row $row_number: One or more required fields are empty.";
                    $total_failed++;
                    continue;
                }

                // Generate password
                $raw_password = generate_random_password();
                $password_hash = password_hash($raw_password, PASSWORD_DEFAULT);

                // Check for existing user (username or staff_id or email) - simplified check for bulk upload
                // Check for existing user (username or email)
                $sql_check = "SELECT user_id FROM users WHERE username = ? OR email = ?";
                $result_check = execute_query($mysqli, $sql_check, 'ss', [$username, $email]);

                if ($result_check && $result_check->num_rows > 0) {
                    $failed_rows[] = "Row $row_number: User with Staff ID/Username or Email already exists.";
                    $total_failed++;
                    continue;
                }

                // Bind and execute insert
                $stmt->bind_param('ssssss', $username, $password_hash, $full_name, $grade, $staff_id, $email);
                
                if ($stmt->execute()) {
                    $total_success++;
                    // NOTE: For bulk upload, we will log the password to a file for the admin to handle,
                    // as sending a large number of emails synchronously is not ideal.
                    $log_entry = "SUCCESS: User $username created. Password: $raw_password. Email: $email\n";
                    file_put_contents(__DIR__ . '/bulk_passwords.log', $log_entry, FILE_APPEND);
                } else {
                    $failed_rows[] = "Row $row_number: Database error: " . $stmt->error;
                    $total_failed++;
                }
            }

            $stmt->close();
            
            if ($total_failed > 0) {
                $mysqli->rollback();
                $error = "Bulk upload failed. No users were added. See details below.";
            } else {
                $mysqli->commit();
                $success = "Bulk upload successful! $total_success test-takers were created. Temporary passwords have been logged to **admin/bulk_passwords.log** for manual distribution.";
            }

            fclose($handle);
        }
    }
}

include 'includes/header.php';
?>
        
        <?php if ($error): ?>
            <p class="error"><?php echo $error; ?></p>
        <?php endif; ?>
        <?php if ($success): ?>
            <p class="success"><?php echo $success; ?></p>
        <?php endif; ?>

        <p>Upload a CSV file with the following columns (in order):</p>
        <p><strong>Full Name, Staff ID/Username, Email, Grade</strong></p>
        <p>Example Row: <code>John Doe,JD1234,john.doe@example.com,RT</code></p>
        <p>
            <a href="<?php echo BASE_URL; ?>bulk_upload_template.csv" download="bulk_upload_template.csv" class="btn btn-secondary">Download Template CSV</a>
        </p>
        
        <form method="POST" action="bulk_upload.php" enctype="multipart/form-data">
            <div class="form-group">
                <label for="csv_file">Select CSV File:</label>
                <input type="file" id="csv_file" name="csv_file" accept=".csv" required>
            </div>
            <button type="submit">Upload and Create Users</button>
        </form>

        <?php if (!empty($failed_rows)): ?>
            <h2>Failed Rows (<?php echo $total_failed; ?>)</h2>
            <div class="error">
                <ul>
                    <?php foreach ($failed_rows as $row_error): ?>
                        <li><?php echo htmlspecialchars($row_error); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>
<?php
include 'includes/footer.php';
?>
