<?php
// bulk_upload_questions.php - Admin Bulk Question Upload

require_once '../db.php';
require_admin();

$page_title = 'Bulk Question Upload';
$error = '';
$success = '';

// Fetch all exams for the dropdown
$sql_exams = "SELECT exam_id, exam_title FROM exams ORDER BY exam_title";
$exams_result = execute_query($mysqli, $sql_exams);
$exams = $exams_result ? $exams_result->fetch_all(MYSQLI_ASSOC) : [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $exam_id = (int)($_POST['exam_id'] ?? 0);

    if ($exam_id <= 0) {
        $error = 'Please select an exam to assign the questions to.';
    } elseif (empty($_FILES['csv_file']['tmp_name'])) {
        $error = 'Please upload a CSV file.';
    } elseif ($_FILES['csv_file']['error'] !== UPLOAD_ERR_OK) {
        $error = 'File upload failed with error code: ' . $_FILES['csv_file']['error'];
    } else {
        $file_handle = fopen($_FILES['csv_file']['tmp_name'], 'r');
        if ($file_handle === FALSE) {
            $error = 'Could not open the uploaded file.';
        } else {
            // Skip header row
            fgetcsv($file_handle);

            $imported_count = 0;
            $failed_count = 0;
            $question_types = ['MCQ', 'Theory', 'Fill-in-the-gaps'];

            $mysqli->begin_transaction();

            while (($data = fgetcsv($file_handle, 1000, ",")) !== FALSE) {
                // Expected CSV columns:
                // 0: Question Type (MCQ, Theory, Fill-in-the-gaps)
                // 1: Question Text
                // 2: Points
                // 3: Option A (for MCQ)
                // 4: Option B (for MCQ)
                // 5: Option C (for MCQ)
                // 6: Option D (for MCQ)
                // 7: Correct Option (A, B, C, D for MCQ)
                // 8: Correct Answer (for Fill-in-the-gaps)
                // 9: Expected Keywords (for Theory)

                $question_type = trim($data[0] ?? '');
                $question_text = trim($data[1] ?? '');
                $points = (float)($data[2] ?? 1.00);

                if (empty($question_text) || !in_array($question_type, $question_types)) {
                    $failed_count++;
                    continue;
                }

                $option_a = $data[3] ?? null;
                $option_b = $data[4] ?? null;
                $option_c = $data[5] ?? null;
                $option_d = $data[6] ?? null;
                $correct_option = $data[7] ?? null;
                $correct_answer = $data[8] ?? null;
                $expected_keywords = $data[9] ?? null;

                $sql_insert = "INSERT INTO exam_questions (exam_id, question_type, question_text, option_a, option_b, option_c, option_d, correct_option, correct_answer, expected_keywords, points) 
                               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
                
                $insert_id = execute_insert($mysqli, $sql_insert, 'issssssssds', [
                    $exam_id, 
                    $question_type, 
                    $question_text, 
                    $option_a, 
                    $option_b, 
                    $option_c, 
                    $option_d, 
                    $correct_option, 
                    $correct_answer, 
                    $expected_keywords,
                    $points
                ]);

                if ($insert_id) {
                    $imported_count++;
                } else {
                    $failed_count++;
                }
            }

            fclose($file_handle);

            if ($failed_count === 0) {
                $mysqli->commit();
                $success = "Successfully imported $imported_count questions.";
            } else {
                $mysqli->rollback();
                $error = "Import failed. $imported_count questions imported, $failed_count failed. Please check the CSV format.";
            }
        }
    }
}

include 'includes/header.php';
?>

<div class="container">
    <h2><?php echo $page_title; ?></h2>

    <?php if ($error): ?>
        <p class="error"><?php echo $error; ?></p>
    <?php endif; ?>
    <?php if ($success): ?>
        <p class="success"><?php echo $success; ?></p>
    <?php endif; ?>

    <p>Use this form to upload a CSV file containing multiple questions.</p>
    <p>
        <a href="../bulk_upload_questions_template.csv" class="btn btn-info" download>Download Sample CSV Template</a>
    </p>

    <form method="POST" enctype="multipart/form-data">
        <div class="form-group">
            <label for="exam_id">Assign Questions to Exam:</label>
            <select id="exam_id" name="exam_id" required>
                <option value="">Select Exam</option>
                <?php foreach ($exams as $exam): ?>
                    <option value="<?php echo $exam['exam_id']; ?>" <?php echo (($_POST['exam_id'] ?? '') == $exam['exam_id']) ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($exam['exam_title']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="form-group">
            <label for="csv_file">Upload CSV File:</label>
            <input type="file" id="csv_file" name="csv_file" accept=".csv" required>
        </div>

        <input type="submit" value="Upload Questions" class="btn btn-primary">
    </form>
</div>

<?php include 'includes/footer.php'; ?>
