<?php
// admin/edit_question.php - Admin interface for editing a question

require_once '../db.php';
require_admin();

$page_title = 'Edit Question';
$error = '';
$success = '';
$question_types = ['MCQ', 'Theory', 'Fill-in-the-gaps'];

$question_id = (int)($_GET['id'] ?? 0);

if ($question_id === 0) {
    redirect('questions.php', 'error', 'Invalid question ID.');
}

// Fetch all exams for the dropdown
$sql_exams = "SELECT exam_id, exam_title, exam_mode FROM exams ORDER BY exam_title";
$exams_result = execute_query($mysqli, $sql_exams);
$exams = $exams_result ? $exams_result->fetch_all(MYSQLI_ASSOC) : [];

// Fetch existing question data
$sql_fetch = "SELECT * FROM exam_questions WHERE question_id = ?";
$result_fetch = execute_query($mysqli, $sql_fetch, 'i', [$question_id]);

if (!$result_fetch || $result_fetch->num_rows === 0) {
    redirect('questions.php', 'error', 'Question not found.');
}

$question = $result_fetch->fetch_assoc();

// Handle POST request for updating the question
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $exam_id = (int)($_POST['exam_id'] ?? 0);
    $question_type = $_POST['question_type'] ?? '';
    $question_text = trim($_POST['question_text'] ?? '');
    $points = (float)($_POST['points'] ?? 1.00);

    // Fetch the selected exam's mode for validation
    $exams_map = [];
    foreach ($exams as $exam) {
        $exams_map[$exam['exam_id']] = $exam['exam_mode'];
    }
    $selected_exam_mode = $exams_map[$exam_id] ?? null;

    // Validation
    if (empty($question_text) || $exam_id <= 0 || !in_array($question_type, $question_types)) {
        $error = 'Please fill in the question text, select an exam, and a valid question type.';
    } elseif ($selected_exam_mode && $selected_exam_mode !== 'Mixed' && $selected_exam_mode !== $question_type) {
        $error = "The selected exam is in '{$selected_exam_mode}' mode and only allows '{$selected_exam_mode}' questions.";
    } else {
        $option_a = $_POST['option_a'] ?? null;
        $option_b = $_POST['option_b'] ?? null;
        $option_c = $_POST['option_c'] ?? null;
        $option_d = $_POST['option_d'] ?? null;
        $correct_option = $_POST['correct_option'] ?? null;
        $correct_answer = $_POST['correct_answer'] ?? null;
        $expected_keywords = $_POST['expected_keywords'] ?? null;

        $sql_update = "UPDATE exam_questions SET 
                       exam_id = ?, 
                       question_type = ?, 
                       question_text = ?, 
                       option_a = ?, 
                       option_b = ?, 
                       option_c = ?, 
                       option_d = ?, 
                       correct_option = ?, 
                       correct_answer = ?, 
                       expected_keywords = ?, 
                       points = ?
                       WHERE question_id = ?";
        
        $affected_rows = execute_non_query($mysqli, $sql_update, 'issssssssdsi', [
            $exam_id, 
            $question_type, 
            $question_text, 
            $option_a, 
            $option_b, 
            $option_c, 
            $option_d, 
            $correct_option, 
            $correct_answer, 
            $expected_keywords,
            $points,
            $question_id
        ]);

        if ($affected_rows !== false) {
            $success = "Question updated successfully.";
            // Re-fetch the updated data
            $result_fetch = execute_query($mysqli, $sql_fetch, 'i', [$question_id]);
            $question = $result_fetch->fetch_assoc();
        } else {
            $error = 'Error updating question.';
        }
    }
}

include 'includes/header.php';
?>

<div class="mb-20">
    <a href="<?php echo BASE_URL; ?>admin/questions.php"><button>Back to Question Bank</button></a>
</div>

<h2>Edit Question (ID: <?php echo $question_id; ?>)</h2>
<?php if ($error): ?>
    <p class="error"><?php echo $error; ?></p>
<?php endif; ?>
<?php if ($success): ?>
    <p class="success"><?php echo $success; ?></p>
<?php endif; ?>
<form method="POST" action="edit_question.php?id=<?php echo $question_id; ?>">
    <div class="form-group">
        <label for="exam_id">Assign to Exam:</label>
        <select id="exam_id" name="exam_id" onchange="updateQuestionTypes()" required>
            <option value="">Select Exam</option>
            <?php foreach ($exams as $exam): ?>
                <option value="<?php echo $exam['exam_id']; ?>" 
                        data-mode="<?php echo htmlspecialchars($exam['exam_mode']); ?>"
                        <?php echo (($question['exam_id'] ?? '') == $exam['exam_id']) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($exam['exam_title']); ?> (Mode: <?php echo htmlspecialchars($exam['exam_mode']); ?>)
                </option>
            <?php endforeach; ?>
        </select>
    </div>
    <div class="form-group">
        <label for="question_type">Question Type:</label>
        <select id="question_type" name="question_type" onchange="toggleQuestionFields()" required>
            <option value="">Select Type</option>
            <?php foreach ($question_types as $type): ?>
                <option value="<?php echo $type; ?>" class="question-type-option" data-type="<?php echo $type; ?>" <?php echo (($question['question_type'] ?? '') === $type) ? 'selected' : ''; ?>>
                    <?php echo $type; ?>
                </option>
            <?php endforeach; ?>
        </select>
    </div>
    <div class="form-group">
        <label for="question_text">Question Text:</label>
        <textarea id="question_text" name="question_text" required><?php echo htmlspecialchars($question['question_text'] ?? ''); ?></textarea>
    </div>
    <div class="form-group">
        <label for="points">Points:</label>
        <input type="number" id="points" name="points" min="0.01" step="0.01" value="<?php echo htmlspecialchars($question['points'] ?? 1.00); ?>" required>
    </div>

    <!-- MCQ Fields -->
    <div id="mcq_fields" style="display: none; border: 1px solid #ccc; padding: 15px; margin-bottom: 15px;">
        <h4 style="margin-top: 0;">MCQ Options</h4>
        <div class="form-group">
            <label for="option_a">Option A:</label>
            <input type="text" id="option_a" name="option_a" value="<?php echo htmlspecialchars($question['option_a'] ?? ''); ?>">
        </div>
        <div class="form-group">
            <label for="option_b">Option B:</label>
            <input type="text" id="option_b" name="option_b" value="<?php echo htmlspecialchars($question['option_b'] ?? ''); ?>">
        </div>
        <div class="form-group">
            <label for="option_c">Option C:</label>
            <input type="text" id="option_c" name="option_c" value="<?php echo htmlspecialchars($question['option_c'] ?? ''); ?>">
        </div>
        <div class="form-group">
            <label for="option_d">Option D:</label>
            <input type="text" id="option_d" name="option_d" value="<?php echo htmlspecialchars($question['option_d'] ?? ''); ?>">
        </div>
        <div class="form-group">
            <label for="correct_option">Correct Option:</label>
            <select id="correct_option" name="correct_option">
                <option value="">Select Correct Option</option>
                <option value="A" <?php echo (($question['correct_option'] ?? '') === 'A') ? 'selected' : ''; ?>>A</option>
                <option value="B" <?php echo (($question['correct_option'] ?? '') === 'B') ? 'selected' : ''; ?>>B</option>
                <option value="C" <?php echo (($question['correct_option'] ?? '') === 'C') ? 'selected' : ''; ?>>C</option>
                <option value="D" <?php echo (($question['correct_option'] ?? '') === 'D') ? 'selected' : ''; ?>>D</option>
            </select>
        </div>
    </div>

    <!-- Theory/Fill-in-the-gaps Fields -->
    <div id="theory_fields" style="display: none; border: 1px solid #ccc; padding: 15px; margin-bottom: 15px;">
        <h4 style="margin-top: 0;">Answer Details</h4>
        <div class="form-group">
            <label for="correct_answer">Correct Answer (for Fill-in-the-gaps):</label>
            <input type="text" id="correct_answer" name="correct_answer" value="<?php echo htmlspecialchars($question['correct_answer'] ?? ''); ?>">
        </div>
        <div class="form-group">
            <label for="expected_keywords">Expected Keywords (for Theory - comma separated):</label>
            <textarea id="expected_keywords" name="expected_keywords"><?php echo htmlspecialchars($question['expected_keywords'] ?? ''); ?></textarea>
        </div>
    </div>

    <script>
        const examsMap = <?php echo json_encode($exams_map); ?>;

        function toggleQuestionFields() {
            const type = document.getElementById('question_type').value;
            const mcqFields = document.getElementById('mcq_fields');
            const theoryFields = document.getElementById('theory_fields');

            mcqFields.style.display = 'none';
            theoryFields.style.display = 'none';

            if (type === 'MCQ') {
                mcqFields.style.display = 'block';
            } else if (type === 'Theory' || type === 'Fill-in-the-gaps') {
                theoryFields.style.display = 'block';
            }
        }

        function updateQuestionTypes() {
            const examSelect = document.getElementById('exam_id');
            const questionTypeSelect = document.getElementById('question_type');
            const selectedExamId = examSelect.value;
            const selectedExamMode = examsMap[selectedExamId];
            
            // Show/hide question types based on exam mode
            document.querySelectorAll('.question-type-option').forEach(option => {
                const type = option.getAttribute('data-type');
                if (!selectedExamId || selectedExamMode === 'Mixed' || selectedExamMode === type) {
                    option.style.display = 'block';
                    option.disabled = false;
                } else {
                    option.style.display = 'none';
                    option.disabled = true;
                }
            });
        }

        // Call on page load to set initial state
        document.addEventListener('DOMContentLoaded', function() {
            toggleQuestionFields();
            updateQuestionTypes();
        });
    </script>

    <input type="submit" value="Update Question">
</form>
<?php
include 'includes/footer.php';
?>
