<?php
// finalize_upload.php - Endpoint to mark the video upload as complete and trigger automatic processing.
// This is called by the Web Worker logic after all chunks are uploaded.

require_once '../db.php';
require_once __DIR__ . '/video_processor.php';

// Set headers for CORS and to indicate JSON response
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');

// Check for POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405); // Method Not Allowed
    echo json_encode(['success' => false, 'message' => 'Only POST requests are allowed.']);
    exit;
}

// Check for required parameters
if (!isset($_POST['session_id'])) {
    http_response_code(400); // Bad Request
    echo json_encode(['success' => false, 'message' => 'Missing required parameter (session_id).']);
    exit;
}

$session_id = (int)$_POST['session_id'];

// 1. Update the student_exam_sessions table
// Set video_upload_complete to TRUE
$sql_update = "UPDATE student_exam_sessions SET video_upload_complete = TRUE WHERE session_id = ?";
$affected_rows = execute_non_query($mysqli, $sql_update, 'i', [$session_id]);

if ($affected_rows === false) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database error during finalization.']);
    exit;
}

if ($affected_rows === 0) {
    http_response_code(404);
    echo json_encode(['success' => false, 'message' => 'Exam session not found or already finalized.']);
    exit;
}

// 2. Trigger automatic video processing in background
// This prevents the finalization call from blocking while FFmpeg processes the video
error_log("Finalize Upload: Starting automatic video processing for session $session_id");

// Start background processing
process_video_background($session_id);

// Success response (processing happens in background)
echo json_encode([
    'success' => true,
    'message' => 'Video upload finalized and automatic processing started for session ' . $session_id . '.',
    'session_id' => $session_id,
    'processing_status' => 'Video processing started in background'
]);

$mysqli->close();
?>
