// upload_worker.js - Web Worker for parallel video chunk uploads

self.onmessage = function(e) {
    const { chunk, chunkIndex, sessionId, uploadUrl, finalizeUrl } = e.data;

    if (chunk === 'FINALIZE') {
        // This is the finalization signal, sent after all chunks are processed
        fetch(finalizeUrl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `session_id=${sessionId}`
        })
        .then(response => response.json())
        .then(data => {
            self.postMessage({ status: 'finalized', success: data.success, message: data.message });
        })
        .catch(error => {
            self.postMessage({ status: 'finalized', success: false, message: `Finalization failed: ${error.message}` });
        });
        return;
    }

    // 1. Prepare FormData for the chunk
    const formData = new FormData();
    formData.append('session_id', sessionId);
    formData.append('chunk_index', chunkIndex);
    // The chunk is a Blob (video data)
    formData.append('video_chunk', chunk, `chunk_${sessionId}_${chunkIndex}.webm`); 

    // 2. Perform the upload using Fetch API
    // The uploadUrl is now a relative path, which is resolved by the browser
    fetch(uploadUrl, {
        method: 'POST',
        body: formData,
        // Important: Do not set 'Content-Type' header for FormData, 
        // the browser will set it automatically with the correct boundary.
    })
    .then(response => {
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        return response.json();
    })
    .then(data => {
        // 3. Send success message back to the main thread
        self.postMessage({ 
            status: 'complete', 
            success: data.success, 
            chunkIndex: chunkIndex, 
            message: data.message 
        });
    })
    .catch(error => {
        // 4. Send error message back to the main thread
        self.postMessage({ 
            status: 'error', 
            success: false, 
            chunkIndex: chunkIndex, 
            message: `Upload failed: ${error.message}` 
        });
    });
};
