<?php
// video_processor.php - Automatic video chunk stitching utility

require_once __DIR__ . '/../db.php';

/**
 * Automatically processes video chunks for a session and creates a single MP4 file.
 * 
 * @param int $session_id The session ID to process
 * @param mysqli $mysqli Database connection
 * @return array Result array with 'success' (bool), 'message' (string), and 'video_path' (string)
 */
function process_video_chunks($session_id, $mysqli)
{
    $result = [
        'success' => false,
        'message' => '',
        'video_path' => null
    ];

    try {
        // 1. Fetch session and student data
        $sql_session = "SELECT ses.student_id, ses.public_session_id, u.username 
                        FROM student_exam_sessions ses
                        JOIN users u ON ses.student_id = u.user_id
                        WHERE ses.session_id = ?";
        $session_result = execute_query($mysqli, $sql_session, 'i', [$session_id]);

        if (!$session_result || $session_result->num_rows === 0) {
            $result['message'] = 'Session not found.';
            error_log("Video Processing Error: Session $session_id not found.");
            return $result;
        }

        $session_data = $session_result->fetch_assoc();
        $username = $session_data['username'];

        // 2. Fetch all video chunks for this session
        $sql_chunks = "SELECT file_path, chunk_index 
                       FROM exam_video_chunks 
                       WHERE session_id = ? 
                       ORDER BY chunk_index ASC";
        $chunks_result = execute_query($mysqli, $sql_chunks, 'i', [$session_id]);

        if (!$chunks_result || $chunks_result->num_rows === 0) {
            $result['message'] = 'No video chunks found for this session.';
            error_log("Video Processing Warning: No chunks found for session $session_id.");
            return $result;
        }

        $chunks = [];
        while ($row = $chunks_result->fetch_assoc()) {
            $chunks[] = $row;
        }

        // 3. Prepare output directory and file paths
        $base_dir = PROJECT_ROOT_DIR;
        $output_dir = $base_dir . 'videos/';

        // Ensure the output directory exists
        if (!is_dir($output_dir)) {
            mkdir($output_dir, 0777, true);
        }

        $temp_list_file = $output_dir . 'list_' . $session_id . '.txt';
        $output_file_name = $username . '_' . $session_id . '.mp4';
        $output_file_path = $output_dir . $output_file_name;

        // 4. Create FFmpeg concat list file
        $list_content = '';
        foreach ($chunks as $chunk) {
            $absolute_chunk_path = $chunk['file_path'];

            // Verify chunk file exists
            if (!file_exists($absolute_chunk_path)) {
                error_log("Video Processing Warning: Chunk file not found: $absolute_chunk_path");
                continue;
            }

            // Escape single quotes in the path for FFmpeg
            $escaped_path = str_replace("'", "'\\''", $absolute_chunk_path);
            $list_content .= "file '" . $escaped_path . "'\n";
        }

        if (empty($list_content)) {
            $result['message'] = 'No valid video chunks found.';
            error_log("Video Processing Error: No valid chunks for session $session_id.");
            return $result;
        }

        file_put_contents($temp_list_file, $list_content);

        // 5. Check for FFmpeg
        $ffmpeg_binary = trim(shell_exec('which ffmpeg 2>/dev/null'));
        if (empty($ffmpeg_binary)) {
            $result['message'] = 'FFmpeg not found on server.';
            error_log("Video Processing Error: FFmpeg binary not found.");
            unlink($temp_list_file);
            return $result;
        }

        // 6. Execute FFmpeg command
        // Use -f concat with -safe 0 for absolute paths
        // -c copy for fast stream copy without re-encoding
        // Redirect stderr to capture errors
        $ffmpeg_command = sprintf(
            "%s -f concat -safe 0 -i %s -c copy -y %s 2>&1",
            escapeshellarg($ffmpeg_binary),
            escapeshellarg($temp_list_file),
            escapeshellarg($output_file_path)
        );

        $output = [];
        $exit_code = 0;
        exec($ffmpeg_command, $output, $exit_code);

        // 7. Clean up temporary list file
        unlink($temp_list_file);

        // 8. Verify output file was created
        if (file_exists($output_file_path) && filesize($output_file_path) > 0) {
            // Update the database with the final video path
            $relative_path = 'videos/' . $output_file_name;
            $sql_update = "UPDATE student_exam_sessions 
                          SET proctoring_video_path = ? 
                          WHERE session_id = ?";
            execute_non_query($mysqli, $sql_update, 'si', [$relative_path, $session_id]);

            $result['success'] = true;
            $result['message'] = 'Video processed successfully.';
            $result['video_path'] = $relative_path;

            error_log("Video Processing Success: Session $session_id processed to $output_file_name");
        } else {
            $result['message'] = 'FFmpeg failed to create output file.';
            $ffmpeg_output = implode("\n", $output);
            error_log("Video Processing Error: FFmpeg failed for session $session_id. Exit code: $exit_code. Output: " . substr($ffmpeg_output, 0, 500));
        }

    } catch (Exception $e) {
        $result['message'] = 'Exception during video processing: ' . $e->getMessage();
        error_log("Video Processing Exception: " . $e->getMessage());
    }

    return $result;
}

/**
 * Process video in background (non-blocking).
 * This function spawns a background process to handle video stitching.
 * 
 * @param int $session_id The session ID to process
 * @return bool True if background process started successfully
 */
function process_video_background($session_id)
{
    $script_path = PROJECT_ROOT_DIR . 'proctor/process_video_background.php';
    $log_file = PROJECT_ROOT_DIR . 'videos/processing_log_' . $session_id . '.txt';

    // Build the command to run in background
    // Build the command to run in background
    $script_path_escaped = escapeshellarg($script_path);
    $log_file_escaped = escapeshellarg($log_file);
    $session_id_int = (int) $session_id;

    if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
        // Windows: Use start /B to run in background without opening a new window
        // Note: This requires the PHP CLI executable to be in the system PATH
        $command = sprintf(
            'start /B php %s %d > %s 2>&1',
            $script_path_escaped,
            $session_id_int,
            $log_file_escaped
        );
    } else {
        // Linux/Unix: Use nohup and & to run in background
        $command = sprintf(
            'nohup php %s %d > %s 2>&1 &',
            $script_path_escaped,
            $session_id_int,
            $log_file_escaped
        );
    }

    // Execute in background
    exec($command);

    error_log("Video Processing: Background process started for session $session_id");
    return true;
}
?>