<?php
// start_exam.php - Student Exam Selection and Start Page

require_once '../db.php';
require_student();

$student_id = get_current_user_id();
$student_grade = get_current_user_grade();
$student_staff_id = get_current_user_staff_id();
$student_username = get_current_username();
$error = '';

// Handle Start Exam Request
if (isset($_POST['start_exam']) && isset($_POST['exam_id'])) {
    $exam_id = (int)$_POST['exam_id'];

    // 1. Check if ANY session (submitted or not) exists for this student and exam
    $sql_existing_session = "SELECT session_id, is_submitted FROM student_exam_sessions WHERE student_id = ? AND exam_id = ?";
    $result_existing = execute_query($mysqli, $sql_existing_session, 'ii', [$student_id, $exam_id]);

    if ($result_existing && $result_existing->num_rows > 0) {
        $existing_session = $result_existing->fetch_assoc();
        
        if ($existing_session['is_submitted']) {
            // Student has already submitted this exam, fetch and display results
            $session_id = $existing_session['session_id'];
            
            $sql_result = "
                SELECT 
                    E.exam_title, 
                    E.total_questions, 
                    SES.total_score,
                    E.duration_minutes
                FROM 
                    student_exam_sessions SES
                JOIN 
                    exams E ON SES.exam_id = E.exam_id
                WHERE 
                    SES.session_id = ?
            ";
            $result_data = execute_query($mysqli, $sql_result, 'i', [$session_id]);
            $exam_result = $result_data ? $result_data->fetch_assoc() : null;

            if ($exam_result) {
                // Store result data in session to display on the page
                $_SESSION['completed_exam_result'] = $exam_result;
            }
            
            // Redirect to the same page to clear POST data and show result message
            redirect('student/start_exam.php');
        } else {
            // Resume existing active session
            redirect('student/camera_test.php?session_id=' . $existing_session['session_id']);
        }
    } else {
        // 2. Create a new exam session
        // Generate the public session ID (KWIRS/1120 format)
        // Note: The public_session_id is now only used for display/tracking, not as a unique key.
        $public_session_id = $student_username;
        if (empty($public_session_id)) {
            // Set an error message and redirect to prevent further execution
            $_SESSION['error_message'] = 'Your account is missing a username, which is required to start the exam.';
            redirect('student/start_exam.php');
        }

        $sql_insert_session = "INSERT INTO student_exam_sessions (student_id, exam_id, public_session_id, start_time) VALUES (?, ?, ?, NOW())";
        $session_id = execute_insert($mysqli, $sql_insert_session, 'iis', [$student_id, $exam_id, $public_session_id]);

        if ($session_id) {
            redirect('student/camera_test.php?session_id=' . $session_id);
        } else {
            // This error will now primarily be caused by the new UNIQUE KEY (student_id, exam_id)
            $_SESSION['error_message'] = 'Error starting new exam session. You may have already attempted this exam.';
            redirect('student/start_exam.php');
        }
    }
}

// Check if student has a grade assigned
if (empty($student_grade)) {
    $error = 'Your account does not have an assigned grade. Please contact your administrator.';
    $exams = [];
} else {
    // Fetch all active exams for the student's grade
    $sql_exams = "SELECT E.exam_id, E.exam_title, E.duration_minutes, E.total_questions, E.exam_mode, E.target_grade AS grade_name
                  FROM exams E
                  WHERE E.is_active = TRUE AND E.target_grade = ?
                  ORDER BY E.exam_title";
    $result_exams = execute_query($mysqli, $sql_exams, 's', [$student_grade]);
    $exams = $result_exams ? $result_exams->fetch_all(MYSQLI_ASSOC) : [];
}

// Handle flash messages from session
if (isset($_SESSION['error_message'])) {
    $error = $_SESSION['error_message'];
    unset($_SESSION['error_message']);
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Start Exam - <?php echo PROJECT_NAME; ?></title>
    <link rel="stylesheet" href="<?php echo MAIN_CSS_PATH; ?>">
</head>
<body>
    <div class="container admin-container">
        <div class="header">
            <h1>Welcome, <?php echo htmlspecialchars($_SESSION['full_name'] ?? 'Student'); ?></h1>
            <a href="<?php echo BASE_URL; ?>logout.php">Logout</a>
        </div>
        <h2>Available Exams (Grade: <?php echo htmlspecialchars($student_grade ?? 'N/A'); ?>)</h2>

	        <?php if ($error): ?>
	            <p class="error"><?php echo $error; ?></p>
	        <?php endif; ?>
	
	        <?php if (isset($_SESSION['completed_exam_result'])): 
	            $result = $_SESSION['completed_exam_result'];
	            unset($_SESSION['completed_exam_result']);
	        ?>
	            <div class="success" style="border: 1px solid #4CAF50; padding: 15px; margin-bottom: 20px; border-radius: 5px;">
	                <h3>Exam Completed: <?php echo htmlspecialchars($result['exam_title']); ?></h3>
	                <p><strong>Status:</strong> Already Submitted</p>
	                <p><strong>Your Score:</strong> <?php echo htmlspecialchars(sprintf("%.2f", $result['total_score'])); ?></p>
	                <p><strong>Total Questions:</strong> <?php echo htmlspecialchars($result['total_questions']); ?></p>
	                <p><strong>Duration (min):</strong> <?php echo htmlspecialchars($result['duration_minutes']); ?></p>
	                <p>You cannot attempt this exam again.</p>
	            </div>
	        <?php endif; ?>
	
	        <?php if (empty($exams)): ?>
            <p>No active exams are currently available for your grade (<?php echo htmlspecialchars($student_grade ?? 'N/A'); ?>).</p>
        <?php else: ?>
            <table>
                <thead>
                    <tr>
                        <th>Exam Title</th>
                        <th>Target Grade</th>
                        <th>Mode</th>
                        <th>Questions</th>
                        <th>Duration (min)</th>
                        <th>Action</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($exams as $exam): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($exam['exam_title']); ?></td>
                            <td><?php echo htmlspecialchars($exam['grade_name']); ?></td>
                            <td><?php echo htmlspecialchars($exam['exam_mode']); ?></td>
                            <td><?php echo htmlspecialchars($exam['total_questions']); ?></td>
                            <td><?php echo htmlspecialchars($exam['duration_minutes']); ?></td>
                            <td>
                                <form method="POST" action="start_exam.php">
                                    <input type="hidden" name="exam_id" value="<?php echo $exam['exam_id']; ?>">
                                    <button type="submit" name="start_exam">Start Exam</button>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
</body>
</html>
<?php $mysqli->close(); ?>
